/*
 * @(#)OMLink.java  1.0  4. Februar 2004
 *
 * Copyright (c) 2003 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */

package ch.hslu.cm.oo.objectmodel;

import ch.hslu.cm.simulation.*;
import java.io.IOException;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;
import java.beans.*;
import java.util.*;
/**
 * Simulates the structure and behavior of a link between two objects.
 * 
 * 
 * @author Werner Randelshofer
 * @version 1.0 4. Februar 2004  Created.
 */
public class OMLink extends AbstractRelationship implements PropertyChangeListener {
    protected PropertyChangeSupport changeSupport;
    protected String memberLabel = "member";
    protected boolean isMemberTraversable = true;
    protected OMAssociation association;
    
    /** Creates a new instance. */
    public OMLink() {
        changeSupport = new PropertyChangeSupport(this);
    }
    
    public void setAssociation(OMAssociation newValue) {
        if (association != null) {
            association.removePropertyChangeListener(this);
            association.removeSimulatedObjectListener(this);
        }
        
        OMAssociation oldValue = association;
        association = newValue;
        
        if (association != null) {
            association.addPropertyChangeListener(this);
            association.addSimulatedObjectListener(this);
        }
        
        changeSupport.firePropertyChange("association", oldValue, newValue);
    }
    
    public OMAssociation getAssociation() {
        return association;
    }
    
    public void addPropertyChangeListener(PropertyChangeListener l) {
        changeSupport.addPropertyChangeListener(l);
    }
    public void removePropertyChangeListener(PropertyChangeListener l) {
        changeSupport.removePropertyChangeListener(l);
    }
    
    public OMObject getAssociate(OMObject startOrEnd) {
        return (OMObject) getConnected(startOrEnd);
    }
    
    public void setMemberLabel(String newValue) {
        String oldValue = memberLabel;
        memberLabel = newValue;
        changeSupport.firePropertyChange("memberLabel", oldValue, newValue);
    }
    public String getMemberLabel() {
        return (association == null) ? memberLabel : association.getMemberLabel();
    }
    
    public void setMemberTraversable(boolean newValue) {
        boolean oldValue = isMemberTraversable;
        isMemberTraversable = newValue;
        changeSupport.firePropertyChange("memberTraversable", oldValue, newValue);
    }
    public boolean isMemberTraversable() {
        return (association == null) ? isMemberTraversable : association.isMemberTraversable();
    }
    
    public int getSimulatedConcept() {
        return ObjectModel.LINK;
    }
    public OMLink clone() {
        OMLink that = (OMLink) super.clone();
        that.changeSupport = new PropertyChangeSupport(that);
        return that;
    }
    
    public boolean canConnect(SimulatedElement start, SimulatedElement end) {
        int concept1 = start.getSimulatedConcept();
        int concept2 = end.getSimulatedConcept();
        
        if (concept1 == ObjectModel.CLASS && concept2 == ObjectModel.CLASS
        || concept1 == ObjectModel.CLASS && concept2 == ObjectModel.OBJECT
        || concept1 == ObjectModel.OBJECT && concept2 == ObjectModel.CLASS) {
            return true;
        } else if (concept1 == ObjectModel.OBJECT && concept2 == ObjectModel.OBJECT) {
            OMObject o1 = (OMObject) start;
            OMObject o2 = (OMObject) end;
            if (o1.getSimulatedClass() == null || o2.getSimulatedClass() == null) {
                return true;
            } else {
                return findAssociationFor(o1, o2) != null;
            }
        }
        return false;
    }
    protected OMAssociation findAssociationFor(OMObject o1, OMObject o2) {
        Collection c = findAssociationsFor(o1, o2);
        
        if (c.size() > 0) {
            for (Iterator i=c.iterator(); i.hasNext(); ) {
                OMAssociation a = (OMAssociation) i.next();
                int selfStartLink = (a == association && o1 == getStart()) ? 1 : 0;
                int selfEndLink = (a == association && o2 == getEnd()) ? 1 : 0;
                if (
                (a.getMemberMultiplicity() < 0 || o1 == getStart() || a.getMemberMultiplicity() > o1.getLinks(a).size() - selfStartLink)
                && (a.getOwnerMultiplicity() < 0 || o2 == getEnd() || a.getOwnerMultiplicity() > o2.getLinks(a).size() - selfEndLink)
                ) {
                    return a;
                }
            }
        }
        return null;
    }
    public LinkedList<OMAssociation> findAssociationsFor(OMObject o1, OMObject o2) {
        OMClass c1 = o1.getSimulatedClass();
        OMClass c2 = o2.getSimulatedClass();
        
        LinkedList<OMAssociation> associations = new LinkedList<OMAssociation>();
        if (c1 != null && c2 != null) {
            associations.addAll(c1.getAllAssociations());
            associations.retainAll(c2.getAllAssociations());
            for (ListIterator i=associations.listIterator(); i.hasNext(); ) {
                OMAssociation a = (OMAssociation) i.next();
                if (a.getOwner().isAssignableFrom(c1)
                && a.getMember().isAssignableFrom(c2)) {
                } else {
                    i.remove();
                }
            }
        }
        return associations;
    }
    
    public boolean canConnect(SimulatedElement start) {
        int concept = start.getSimulatedConcept();
        if (concept == ObjectModel.CLASS) {
            return true;
        } else if (concept == ObjectModel.OBJECT) {
            OMObject o = (OMObject) start;
            OMClass c = o.getSimulatedClass();
            if (c == null) {
                return true;
            }
            return c.getAllAssociations().size() > 0;
        }
        return false;
    }
    
    /**
     * Handles the disconnection of a connection.
     * Override this method to handle this event.
     */
    protected void handleDisconnecting(SimulatedElement start, SimulatedElement end) {
        // maybe we should delete our reference to the association?
    }
    
    /**
     * Handles the connection of a connection.
     * Override this method to handle this event.
     */
    protected void handleConnecting(SimulatedElement start, SimulatedElement end) {
        OMObject o1 = (OMObject) start;
        OMObject o2 = (OMObject) end;
        
        setAssociation(findAssociationFor(o1, o2));
    }
    
    
    public void write(DOMOutput out) throws IOException {
        out.openElement("start");
        out.writeObject(getStart());
        out.closeElement();
        
        out.openElement("end");
        out.addAttribute("name", memberLabel);
        if (isMemberTraversable) {
            out.addAttribute("traversable", isMemberTraversable);
        }
        out.writeObject(getEnd());
        out.closeElement();
        
        out.openElement("association");
        out.writeObject(association);
        out.closeElement();
    }
    public void read(DOMInput in) throws IOException {
        in.openElement("start");
        setStart((SimulatedElement) in.readObject());
        in.closeElement();
        
        in.openElement("end");
        memberLabel = in.getAttribute("name", "member");
        isMemberTraversable = in.getAttribute("traversable", false);
        setEnd((SimulatedElement)  in.readObject());
        in.closeElement();
        
        in.openElement("association");
        setAssociation((OMAssociation) in.readObject());
        in.closeElement();
    }
    
    public void propertyChange(PropertyChangeEvent evt) {
        changeSupport.firePropertyChange(evt.getPropertyName(), evt.getOldValue(), evt.getNewValue());
    }
    
    /**
     * Sent when an element was removed from a simulation.
     */
    public void objectRemoved(SimulatedObjectEvent e) {
        if (e.getObject() == association) fireObjectRequestRemove();
        super.objectRemoved(e);
    }
}
